// utils.js

export function onDomReady(callback) {
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', callback);
    } else {
        callback();
    }
}

export function exposeToWindow(exportsObj) {
    if (typeof window !== 'undefined') {
        const keys = Object.keys(exportsObj);
        for (const key of keys) {
            window[key] = exportsObj[key];
        }
    }
}

export function triggerEvent(el, type, options = {}) {
    const event = new CustomEvent(type, {
        bubbles: true,
        cancelable: true,
        detail: options
    });
    el.dispatchEvent(event);
}

export function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}

export function debounce(el, foo, delay = 300) {
    if (el) {
        clearTimeout(el._timer);
        el._timer = setTimeout(() => {
            foo();
        }, delay);
    } else {
        setTimeout(foo, delay);
    }
}



async function exportTheme() {
    // Загружаем base.css
    const baseUrl = document.querySelector('link[href$="base.css"]').href;
    const baseCss = await fetch(baseUrl).then(r => r.text());

    // Берём CSS с переменными
    const themeCss = document.getElementById('theme').textContent;

    // Объединяем: сначала переменные, потом классы
    const mergedCss = themeCss + "\n\n" + baseCss;

    // Сохраняем как файл
    const blob = new Blob([mergedCss], {
        type: "text/css"
    });
    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = "merged-theme.css";
    link.click();
}

export async function download(data, name = "", type = "text/plain") {
    if (data instanceof Blob) {
        blob = data;
    } else if (data instanceof ArrayBuffer) {
        blob = new Blob([data], {
            type
        });
    } else {
        blob = new Blob([data], {
            type: `${type};charset=utf-8`
        });
    }
    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = name;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    setTimeout(() => URL.revokeObjectURL(link.href), 100);
}